<?php

namespace App\Http\Controllers;

use App\Models\Occurrence;
use App\Models\Attachment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Log;

class OccurrenceController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $query = Occurrence::with(['company', 'type', 'user', 'attachments']);
        if ($user && $user->role !== 'superadmin') {
            $query->whereHas('company', fn($q) => $q->where('created_by', $user->id));
        }

        return $query->get();
    }

    /**
     * Show a page for a specific occurrence type (Chamadas, Emails, Recepcao)
     */
    public function typePage(Request $request)
    {
        $slug = $request->route('slug') ?? $request->get('slug');
        // normalize helper: remove accents and lowercase for robust matching
        $normalize = fn($s) => \Illuminate\Support\Str::ascii(strtolower($s));
        $target = $normalize($slug);

        $occType = null;
        foreach (\App\Models\OccurrenceType::all() as $t) {
            $n = $normalize($t->name);
            // plural-insensitive and substring match (but not reverse substring to avoid false positives)
            $nTrim = rtrim($n, 's');
            $tTrim = rtrim($target, 's');
            if ($n === $target || strpos($n, $target) !== false || $nTrim === $tTrim) {
                $occType = $t;
                break;
            }
        }

        if (! $occType) {
            // If the DB doesn't have occurrence types yet, allow known slugs and create them on demand
            $fallbacks = [
                'chamadas' => 'Chamadas',
                'emails' => 'Emails',
                'recepcao' => 'Recepção',
            ];

            if (isset($fallbacks[$target])) {
                $occType = \App\Models\OccurrenceType::firstOrCreate(['name' => $fallbacks[$target]], ['description' => 'Auto-created']);
            }
        }

        if (! $occType) {
            abort(404);
        }

        $user = Auth::user();
        $query = Occurrence::with(['company', 'user', 'attachments'])->where('occurrence_type_id', $occType->id);

        // Non-superadmin users: employees can see all companies, clients only their company
        if ($user && $user->role === 'client') {
            $query->whereHas('company', fn($q) => $q->where('created_by', $user->id));
        }

        $occurrences = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('occurrences.type', compact('occType', 'occurrences'));
    }

    /**
     * Show the create form for a specific occurrence type
     */
    public function createForType(Request $request)
    {
        // only superadmin and employee can access the create form
        $this->authorize('create', \App\Models\Occurrence::class);

        $slug = $request->route('slug') ?? $request->get('slug');
        $normalize = fn($s) => \Illuminate\Support\Str::ascii(strtolower($s));
        $target = $normalize($slug);
        $occType = null;
        foreach (\App\Models\OccurrenceType::all() as $t) {
            $n = $normalize($t->name);
            if ($n === $target || strpos($n, $target) !== false) {
                $occType = $t;
                break;
            }
        }

        if (! $occType) {
            $fallbacks = [
                'chamadas' => 'Chamadas',
                'emails' => 'Emails',
                'recepcao' => 'Recepção',
            ];

            if (isset($fallbacks[$target])) {
                $occType = \App\Models\OccurrenceType::firstOrCreate(['name' => $fallbacks[$target]], ['description' => 'Auto-created']);
            }
        }

        if (! $occType) {
            abort(404);
        }

        $user = Auth::user();

        // employees and superadmin see all companies; clients see only their company
        if ($user && in_array($user->role, ['superadmin', 'employee'])) {
            $companies = \App\Models\Company::all();
        } else {
            $companies = \App\Models\Company::where('created_by', $user->id)->get();
        }

        return view('occurrences.create', compact('occType', 'companies'));
    }

    /**
     * Show a single occurrence (web page).
     */
    public function show(Occurrence $occurrence)
    {
        $this->authorize('view', $occurrence);

        $occurrence->load(['company','user','attachments','type']);

        return view('occurrences.show', compact('occurrence'));
    }

    public function store(Request $request)
    {
        // ensure the user can create occurrences
        $this->authorize('create', \App\Models\Occurrence::class);

        $data = $request->validate([
            'company_id' => 'required|exists:companies,id',
            'occurrence_type_id' => 'required|exists:occurrence_types,id',
            'title' => 'nullable|string|max:255',
            'details' => 'nullable|string',
            'attachments.*' => 'file|max:5120', // max 5MB each
        ]);

        // set authenticated user as creator
        $user = Auth::user();
        if ($user) {
            $data['user_id'] = $user->id;
        }

        // Ensure newly created occurrences start as 'closed'
        $data['status'] = 'closed';

        $occurrence = Occurrence::create($data);

        // Log incoming and saved type ids to help debug cases where all
        // occurrences appear to be saved as 'Chamadas'. This gives visibility
        // into what the request sent and what was persisted.
        try {
            Log::info('OccurrenceController@store - request occurrence_type_id', ['request' => $data['occurrence_type_id'] ?? null]);
            Log::info('OccurrenceController@store - saved occurrence_type_id', ['saved' => $occurrence->occurrence_type_id ?? null, 'occurrence_id' => $occurrence->id]);
        } catch (\Throwable $e) {
            // swallow logging exceptions
        }

        // Safeguard: if for any reason the created model does not have the
        // requested occurrence_type_id, enforce it and persist. This prevents
        // situations where forms or mass-assignment cause an unexpected type.
        if (isset($data['occurrence_type_id']) && ((int) $occurrence->occurrence_type_id !== (int) $data['occurrence_type_id'])) {
            $occurrence->occurrence_type_id = (int) $data['occurrence_type_id'];
            $occurrence->save();
            try { Log::info('OccurrenceController@store - corrected saved occurrence_type_id', ['saved' => $occurrence->occurrence_type_id, 'occurrence_id' => $occurrence->id]); } catch (\Throwable$e) {}
        }

        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('attachments', 'public');
                Attachment::create([
                    'occurrence_id' => $occurrence->id,
                    'path' => $path,
                    'original_name' => $file->getClientOriginalName(),
                ]);
            }
        }

        // If the request expects JSON (API or AJAX), return JSON as before.
        if ($request->wantsJson() || $request->expectsJson() || $request->ajax()) {
            return response()->json($occurrence->load('attachments'), 201);
        }

        // Otherwise (normal web form submission), redirect to the appropriate
        // occurrence listing page based on user role and occurrence type.
        $occType = $occurrence->type()->first();
        // derive a normalized token from the occurrence type name so we can map
        // English or Portuguese type names to the known route slugs.
        $slug = null;
        if ($occType && isset($occType->name)) {
            $norm = \Illuminate\Support\Str::ascii(strtolower($occType->name));
            if (str_contains($norm, 'cham') || str_contains($norm, 'call')) {
                $slug = 'chamadas';
            } elseif (str_contains($norm, 'email') || str_contains($norm, 'e-mail')) {
                $slug = 'emails';
            } elseif (str_contains($norm, 'recep') || str_contains($norm, 'recept') || str_contains($norm, 'reception')) {
                $slug = 'recepcao';
            } else {
                // fallback: use the same slug generation as views do (dash-separated ascii)
                $slug = \Illuminate\Support\Str::ascii(\Illuminate\Support\Str::slug($occType->name));
            }
        }

        $user = Auth::user();
        if ($user && $user->role === 'employee') {
            if ($slug && \Illuminate\Support\Str::length($slug) > 0) {
                if (Route::has('employee.' . $slug)) {
                    return redirect()->route('employee.' . $slug)->with('message', 'Ocorrência registrada.');
                }
            }

            return redirect()->route('employee.dashboard')->with('message', 'Ocorrência registrada.');
        }

        if ($user && $user->role === 'client') {
            if ($slug && \Illuminate\Support\Str::length($slug) > 0) {
                if (Route::has('client.' . $slug)) {
                    return redirect()->route('client.' . $slug)->with('message', 'Ocorrência registrada.');
                }
            }
            return redirect()->route('company.panel', $occurrence->company_id)->with('message', 'Ocorrência registrada.');
        }

        // Fallback: redirect to the generic occurrences type page
        if ($slug && \Illuminate\Support\Str::length($slug) > 0 && Route::has('occurrences.' . $slug)) {
            return redirect()->route('occurrences.' . $slug)->with('message', 'Ocorrência registrada.');
        }

        return redirect()->route('occurrences.index')->with('message', 'Ocorrência registrada.');
    }

    /**
     * Show edit form for an occurrence.
     */
    public function edit(Occurrence $occurrence)
    {
        $this->authorize('update', $occurrence);

        // ensure we have companies and types available for the form
        $companies = \App\Models\Company::all();
        $types = \App\Models\OccurrenceType::all();

        return view('occurrences.edit', compact('occurrence','companies','types'));
    }

    /**
     * Update an occurrence.
     */
    public function update(Request $request, Occurrence $occurrence)
    {
        $this->authorize('update', $occurrence);

        $data = $request->validate([
            'title' => 'nullable|string|max:255',
            'details' => 'nullable|string',
        ]);

        $occurrence->fill($data);
        // Ensure edited occurrences reset to 'closed'
        $occurrence->status = 'closed';
        $occurrence->save();

        return redirect()->route('occurrences.show', $occurrence->id)->with('message','Ocorrência atualizada.');
    }
}